<?php
namespace app\modules\admin\controllers;

use Yii;
use yii\web\Controller;
use yii\filters\AccessControl;
use app\models\Appointment;
use app\models\User;
use app\models\Service;
use app\models\MasterInfo;
use yii\db\Query;

class ReportsController extends Controller
{
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    [
                        'allow' => true,
                        'roles' => ['@'],
                        'matchCallback' => function ($rule, $action) {
                            return Yii::$app->user->identity->is_admin == 2;
                        }
                    ],
                ],
            ],
        ];
    }

    /**
     * Главная страница отчетов
     */
    public function actionIndex()
    {
        return $this->render('index');
    }

    /**
     * Отчет по количеству заявок
     */
    public function actionAppointments()
    {
        $fromDate = Yii::$app->request->get('from_date', date('Y-m-d', strtotime('-30 days')));
        $toDate = Yii::$app->request->get('to_date', date('Y-m-d'));

        // Статистика по дням
        $dailyStats = Appointment::find()
            ->select(['DATE(start_time) as date', 'COUNT(*) as count', 'SUM(CASE WHEN status = 3 THEN 1 ELSE 0 END) as completed'])
            ->where(['between', 'DATE(start_time)', $fromDate, $toDate])
            ->groupBy(['DATE(start_time)'])
            ->orderBy('date DESC')
            ->asArray()
            ->all();

        // Статистика по статусам
        $statusStats = [
            'total' => Appointment::find()->count(),
            'new' => Appointment::find()->where(['status' => 0])->count(),
            'confirmed' => Appointment::find()->where(['status' => 1])->count(),
            'cancelled' => Appointment::find()->where(['status' => 2])->count(),
            'completed' => Appointment::find()->where(['status' => 3])->count(),
        ];

        return $this->render('appointments', [
            'fromDate' => $fromDate,
            'toDate' => $toDate,
            'dailyStats' => $dailyStats,
            'statusStats' => $statusStats,
        ]);
    }

    /**
     * Отчет по выручке
     */
    public function actionRevenue()
    {
        $fromDate = Yii::$app->request->get('from_date', date('Y-m-d', strtotime('-30 days')));
        $toDate = Yii::$app->request->get('to_date', date('Y-m-d'));

        // Выручка по дням
        $dailyRevenue = Appointment::find()
            ->select([
                'DATE(start_time) as date',
                'COUNT(*) as count',
                'SUM(s.price) as revenue'
            ])
            ->innerJoin('services s', 's.id = appointments.service_id')
            ->where(['between', 'DATE(start_time)', $fromDate, $toDate])
            ->andWhere(['status' => [1, 3]]) // подтвержденные и завершенные
            ->groupBy(['DATE(start_time)'])
            ->orderBy('date DESC')
            ->asArray()
            ->all();

        // Выручка по услугам
        $serviceRevenue = Appointment::find()
            ->select([
                's.name as service_name',
                's.category_id',
                'COUNT(*) as count',
                'SUM(s.price) as revenue'
            ])
            ->innerJoin('services s', 's.id = appointments.service_id')
            ->where(['between', 'DATE(start_time)', $fromDate, $toDate])
            ->andWhere(['status' => [1, 3]])
            ->groupBy(['s.id', 's.name', 's.category_id'])
            ->orderBy('revenue DESC')
            ->asArray()
            ->all();

        // Общая выручка
        $totalRevenue = Appointment::find()
            ->innerJoin('services s', 's.id = appointments.service_id')
            ->where(['between', 'DATE(start_time)', $fromDate, $toDate])
            ->andWhere(['status' => [1, 3]])
            ->sum('s.price');

        return $this->render('revenue', [
            'fromDate' => $fromDate,
            'toDate' => $toDate,
            'dailyRevenue' => $dailyRevenue,
            'serviceRevenue' => $serviceRevenue,
            'totalRevenue' => $totalRevenue,
        ]);
    }

    /**
     * Отчет по загруженности мастеров
     */
    public function actionMastersLoad()
    {
        $fromDate = Yii::$app->request->get('from_date', date('Y-m-d', strtotime('-30 days')));
        $toDate = Yii::$app->request->get('to_date', date('Y-m-d'));

        $masters = User::find()->where(['is_admin' => 1])->all();

        $masterStats = [];
        foreach ($masters as $master) {
            $masterInfo = MasterInfo::find()->where(['user_id' => $master->id])->one();
            if (!$masterInfo) continue;

            $appointmentsCount = Appointment::find()
                ->where(['master_info_id' => $masterInfo->id])
                ->andWhere(['between', 'DATE(start_time)', $fromDate, $toDate])
                ->count();

            $completedCount = Appointment::find()
                ->where(['master_info_id' => $masterInfo->id])
                ->andWhere(['between', 'DATE(start_time)', $fromDate, $toDate])
                ->andWhere(['status' => 3])
                ->count();

            $revenue = Appointment::find()
                ->innerJoin('services s', 's.id = appointments.service_id')
                ->where(['appointments.master_info_id' => $masterInfo->id])
                ->andWhere(['between', 'DATE(start_time)', $fromDate, $toDate])
                ->andWhere(['status' => [1, 3]])
                ->sum('s.price');

            $masterStats[] = [
                'master' => $master,
                'info' => $masterInfo,
                'appointments' => $appointmentsCount,
                'completed' => $completedCount,
                'revenue' => $revenue ?: 0,
            ];
        }

        return $this->render('masters-load', [
            'fromDate' => $fromDate,
            'toDate' => $toDate,
            'masterStats' => $masterStats,
        ]);
    }
}