<?php
namespace app\controllers;

use app\models\Appointment;
use app\models\User;
use Yii;
use yii\web\Controller;
use yii\data\Pagination;
use app\models\Service;
use app\models\ServiceCategory;
use app\models\MasterInfo;
use app\models\MasterService;

class CatalogController extends Controller
{
    /**
     * Главная страница каталога услуг
     */
    public function actionIndex()
    {
        $query = Service::find()
            ->with(['category', 'master'])
            ->where(['is_active' => 1]);

        // Фильтр по категории
        $categoryId = Yii::$app->request->get('category');
        if ($categoryId) {
            $query->andWhere(['category_id' => $categoryId]);
        }

        // Фильтр по поиску
        $search = Yii::$app->request->get('search');
        if ($search) {
            $query->andWhere(['like', 'name', $search]);
        }

        // Фильтр по цене
        $minPrice = Yii::$app->request->get('min_price');
        $maxPrice = Yii::$app->request->get('max_price');
        if ($minPrice) {
            $query->andWhere(['>=', 'price', $minPrice]);
        }
        if ($maxPrice) {
            $query->andWhere(['<=', 'price', $maxPrice]);
        }

        // Фильтр по длительности
        $duration = Yii::$app->request->get('duration');
        if ($duration) {
            $query->andWhere(['<=', 'duration', $duration]);
        }

        // Сортировка
        $sort = Yii::$app->request->get('sort', 'name');
        $order = Yii::$app->request->get('order', 'asc');
        $query->orderBy([$sort => $order == 'asc' ? SORT_ASC : SORT_DESC]);

        // Пагинация
        $countQuery = clone $query;
        $pages = new Pagination(['totalCount' => $countQuery->count()]);
        $pages->pageSize = 12;

        $services = $query->offset($pages->offset)
            ->limit($pages->limit)
            ->all();

        // Все категории для фильтра
        $categories = ServiceCategory::find()
            ->orderBy(['sort' => SORT_ASC, 'name' => SORT_ASC])
            ->all();

        return $this->render('index', [
            'services' => $services,
            'categories' => $categories,
            'pages' => $pages,
            'selectedCategory' => $categoryId,
            'search' => $search,
            'minPrice' => $minPrice,
            'maxPrice' => $maxPrice,
            'duration' => $duration,
            'sort' => $sort,
            'order' => $order,
        ]);
    }

    /**
     * Страница услуги
     */
    public function actionView($id)
    {
        $service = Service::find()
            ->where(['id' => $id, 'is_active' => 1])
            ->with(['category'])
            ->one();

        if (!$service) {
            throw new \yii\web\NotFoundHttpException('Услуга не найдена');
        }

        // Мастера, выполняющие эту услугу
        $masters = User::find()
            ->alias('u')
            ->select(['u.*', 'mi.*', 'ms.price_modifier'])
            ->innerJoin('master_info mi', 'mi.user_id = u.id')
            ->innerJoin('master_services ms', 'ms.master_info_id = mi.id')
            ->where(['ms.service_id' => $service->id])
            ->andWhere(['u.is_admin' => 1])
            ->andWhere(['mi.is_active' => 1])
            ->all();

        // Похожие услуги
        $similarServices = Service::find()
            ->where(['category_id' => $service->category_id, 'is_active' => 1])
            ->andWhere(['!=', 'id', $service->id])
            ->limit(4)
            ->all();

        return $this->render('view', [
            'service' => $service,
            'masters' => $masters,
            'similarServices' => $similarServices,
        ]);
    }

    /**
     * API для проверки доступности мастера по времени
     */
    public function actionCheckAvailability($master_id, $date, $service_id)
    {
        Yii::$app->response->format = \yii\web\Response::FORMAT_JSON;

        $service = Service::findOne($service_id);
        if (!$service) {
            return ['error' => 'Услуга не найдена'];
        }

        $masterInfo = MasterInfo::find()->where(['user_id' => $master_id])->one();
        if (!$masterInfo) {
            return ['error' => 'Мастер не найден'];
        }

        // Получаем занятые записи
        $startOfDay = $date . ' 00:00:00';
        $endOfDay = $date . ' 23:59:59';

        $busyTimes = Appointment::find()
            ->where(['master_info_id' => $masterInfo->id])
            ->andWhere(['between', 'start_time', $startOfDay, $endOfDay])
            ->andWhere(['status' => [0, 1]])
            ->all();

        // Генерируем свободные слоты
        $freeSlots = [];
        $workStart = 9;
        $workEnd = 20;

        $currentTime = strtotime($date . ' ' . $workStart . ':00:00');
        $endTime = strtotime($date . ' ' . $workEnd . ':00:00');

        while ($currentTime + $service->duration * 60 <= $endTime) {
            $slotStart = $currentTime;
            $slotEnd = $currentTime + $service->duration * 60;

            $isFree = true;
            foreach ($busyTimes as $busy) {
                $busyStart = strtotime($busy->start_time);
                $busyEnd = strtotime($busy->end_time);

                if ($slotStart < $busyEnd && $slotEnd > $busyStart) {
                    $isFree = false;
                    break;
                }
            }

            if ($isFree) {
                $freeSlots[] = date('H:i', $slotStart);
            }

            $currentTime += 30 * 60;
        }

        return [
            'free_slots' => $freeSlots,
            'duration' => $service->duration,
        ];
    }
}